import { asyncHandler } from "../../utils/asyncHandler.js";
import { ApiError } from "../../utils/ApiError.js";
import { User } from "../../models/user.model.js";
import { ApiResponse } from "../../utils/ApiResponse.js";
import { registerUserService } from "../../services/auth.service.js";
import { loginUserService } from "../../services/auth.service.js";
import { verifyOTPService } from "../../services/auth.service.js";
import { logoutUserService } from "../../services/auth.service.js";
import { refreshTokenService } from "../../services/auth.service.js";


const registerUser = asyncHandler(async (req, res) => {
  const { user, verifyToken } = await registerUserService(req.body);
  // console.log(user);  

  res.cookie("verify_token", verifyToken, {
    httpOnly: true,
    secure: false, // use true in prod
    sameSite: "Lax",
    maxAge: 15 * 60 * 1000
  });

  return res
    .status(201)
    .setHeader("verificationtoken", verifyToken)
    .json(new ApiResponse(200, { email: user.email, phone: user.phone }, "OTP sent to email"));
});


const verifyOTP = asyncHandler(async (req, res) => {
  const { user, accessToken, refreshToken } = await verifyOTPService(req);

  const cookieOptions = {
    httpOnly: true,
    secure: process.env.NODE_ENV === "production",
    sameSite: "Lax",
    maxAge: 7 * 24 * 60 * 60 * 1000, // 7 days
  };

  return res
    .status(200)
    .clearCookie("verify_token")
    .cookie("accessToken", accessToken, cookieOptions)
    .cookie("refreshToken", refreshToken, cookieOptions)
    .setHeader("accessToken", accessToken)
    .setHeader("refreshToken", refreshToken)
    .json(
      new ApiResponse(
        200,
        { user },
        "OTP verified and user logged in successfully"
      )
    );
});


const loginUser = asyncHandler(async (req, res) => {
  const result = await loginUserService(req);

  // Case 1: User is unverified — OTP flow
  if (result.requiresOTP) {
    const { verifyToken, user } = result;

    // Set verify token in cookie + header
    res
      .cookie("verify_token", verifyToken, {
        httpOnly: true,
        secure: process.env.NODE_ENV === "production",
        sameSite: "Lax",
        maxAge: 15 * 60 * 1000, // 15 minutes
      })
      .setHeader("verificationtoken", verifyToken)
      .status(200)
      .json(
        new ApiResponse(200, { user }, "OTP sent. Please verify to continue.")
      );
    return;
  }

  // Case 2: User is verified — login success
  const { user, accessToken, refreshToken } = result;

  const cookieOptions = {
    httpOnly: true,
    secure: process.env.NODE_ENV === "production",
    sameSite: "Lax",
    maxAge: 7 * 24 * 60 * 60 * 1000, // 7 days
  };

  return res
    .status(200)
    .cookie("accessToken", accessToken, cookieOptions)
    .cookie("refreshToken", refreshToken, cookieOptions)
    .setHeader("accessToken", accessToken)
    .setHeader("refreshToken", refreshToken)
    .json(new ApiResponse(200, { user }, "User logged in successfully"));
});


const logoutUser = asyncHandler(async (req, res) => {
  const refreshToken =
    req.cookies?.refreshToken ||
    req.header("Authorization")?.replace("Bearer ", "");

  if (!refreshToken) {
    throw new ApiError(400, "Refresh token is required for logout");
  }

  await logoutUserService(refreshToken); // ✅ Logic handled in service

  const cookieOptions = {
    httpOnly: true,
    secure: process.env.NODE_ENV === "production",
    sameSite: "Lax",
  };

  return res
    .clearCookie("accessToken", cookieOptions)
    .clearCookie("refreshToken", cookieOptions)
    .status(200)
    .json(new ApiResponse(200, null, "User logged out successfully"));
});


const refreshAccessToken = asyncHandler(async (req, res) => {
  const incomingRefreshToken =
    req.cookies?.refreshToken ||
    req.header("Authorization")?.replace("Bearer ", "");

  if (!incomingRefreshToken) {
    throw new ApiError(401, "Unauthorized: Refresh token missing");
  }

  const { accessToken, refreshToken } = await refreshTokenService(incomingRefreshToken, req);

  const cookieOptions = {
    httpOnly: true,
    secure: process.env.NODE_ENV === "production",
    sameSite: "Lax",
    maxAge: 7 * 24 * 60 * 60 * 1000 // 7 days
  };

  return res
    .status(200)
    .cookie("accessToken", accessToken, cookieOptions)
    .cookie("refreshToken", refreshToken, cookieOptions)
    .setHeader("accessToken", accessToken)
    .setHeader("refreshToken", refreshToken)
    .json(new ApiResponse(200, null, "Tokens refreshed successfully"));
});


const resendOTP = asyncHandler(async (req, res) => {
  const token = req.cookies.verify_token || req.headers["verificationtoken"];

  if (!token) {
    throw new ApiError(401, "Verification token missing");
  }

  const { verifyToken, user } = await resendOTPService(token);

  res.cookie("verify_token", verifyToken, {
    httpOnly: true,
    secure: process.env.NODE_ENV === "production",
    sameSite: "Lax",
    maxAge: 15 * 60 * 1000 // 15 mins
  });

  return res
    .status(200)
    .setHeader("verificationtoken", verifyToken)
    .json(
      new ApiResponse(
        200,
        { email: user.email },
        "New OTP has been sent to your email"
      )
    );
});


const getCurrentUser = asyncHandler(async (req, res) => {
  // Assuming req.user is populated by auth middleware
  const userId = req.user?._id;

  if (!userId) {
    return res.status(401).json(new ApiResponse(401, null, "Unauthorized"));
  }

  const user = await User.findById(userId).select("fullName email phone userType");

  // console.log("Current user:", user);
  if (!user) {
    return res.status(404).json(new ApiResponse(404, null, "User not found"));
  }

  return res.status(200).json(
    new ApiResponse(200, {
      name: user.fullName,
      userType: user.userType
    }, "User fetched successfully")
  );
});


export {
    registerUser,
    resendOTP,
    verifyOTP,
    loginUser,
    logoutUser,
    refreshAccessToken,
    getCurrentUser,
  }