import { User } from "../models/user.model.js";
import { UserAuth } from "../models/userAuth.model.js";
import { Token } from "../models/token.model.js";
import { ApiError } from "../utils/ApiError.js";
import { generateOTP } from "../utils/generateOTP.js"
import { sendOTP } from "../services/emailService.js";
import jwt from "jsonwebtoken";
import { generateAccessAndRefreshTokens } from "../utils/GenerateTokens.js"

export const registerUserService = async ({
  fullName,
  email,
  phone,
  password,
  confirmpassword,
  userType,
  fromGoogle,
  providerId
}) => {
  if (fromGoogle) {
    if ([fullName, email, phone, providerId].some((field) => !field?.trim())) {
      throw new ApiError(400, "All fields are required for Google registration");
    }
  } else {
    if ([fullName, email, phone, password, confirmpassword].some((field) => !field?.trim())) {
      throw new ApiError(400, "All fields are required for local registration");
    }
  }

  // Validations...
  const nameRegex = /^[A-Za-z\s]{2,50}$/;
  if (!nameRegex.test(fullName)) throw new ApiError(400, "Invalid name format");

  const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
  if (!emailRegex.test(email)) throw new ApiError(400, "Invalid email");

  const phoneRegex = /^\d{10}$/;
  if (!phoneRegex.test(phone)) throw new ApiError(400, "Phone must be 10 digits");

  if (!fromGoogle) {
    const passwordRegex = /^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)(?=.*[\W_]).{8,20}$/;
    if (!passwordRegex.test(password)) {
       throw new ApiError(400, "Password must be 8-20 characters long and include uppercase, lowercase, number, and special character.");
    }

    if (password !== confirmpassword) {
      throw new ApiError(400, "Passwords do not match");
    }
  }

  const existingUser = await User.findOne({ $or: [{ email }, { phone }] });
  if (existingUser) throw new ApiError(409, "User with email or phone already exists");

  // 1. Create User
  const user = await User.create({ fullName, email, phone, userType });

  // 2. Create UserAuth
  const userAuth = await UserAuth.create({
    provider: fromGoogle ? "google" : "local",
    providerId: fromGoogle ? providerId : email,
    password: !fromGoogle ? password : undefined,
    userId: user._id
  });

  // 3. Generate OTP + Token
  const otp = generateOTP();
  const verifyToken = jwt.sign({ userId: user._id }, process.env.VERIFY_TOKEN_SECRET, {
    expiresIn: "15m"
  });

  const otpType = fromGoogle ? "phone" : "email";

  await Token.create({
    tokenType: "verification",
    token: verifyToken,
    otp,
    otpType,
    otpExpireAt: new Date(Date.now() + 10 * 60 * 1000),
    isVerified: false,
    isUsed: false,
    userId: user._id,
    authId: userAuth._id,
    expiresAt: new Date(Date.now() + 15 * 60 * 1000)
  });

  await sendOTP(email, otp);

  // const recipient = fromGoogle ? phone : email;

  // await sendOTP(recipient, otp);

  // if (!fromGoogle) {
  //   await sendOTP(email, otp);
  // }
//  console.log(user);
 
  return { user, verifyToken };
};


export const loginUserService = async (req) => {
  const { email, password, userType } = req.body;

  // 1. Validate input
  if ([email, password].some((f) => !f?.trim()) || !userType) {
    throw new ApiError(400, "Email, Password, and userType are required");
  }

  // 2. Find user
  const user = await User.findOne({ email });
  if (!user) {
    throw new ApiError(404, "Invalid credentials");
  }

  // 3. ✅ Check if userType matches
  if (user.userType !== userType) {
    throw new ApiError(
      422,
      `You are registered as a '${user.userType}'. Please use the correct login portal.`
    );
  }

  // 3. Find local UserAuth
  const userAuth = await UserAuth.findOne({
    userId: user._id,
    provider: "local",
  });

  if (!userAuth || !userAuth.isActive) {
    throw new ApiError(403, "You are a google user! try to sign in with Google");
  }

  // 4. Check password
  const isPasswordValid = await userAuth.isPasswordCorrect(password);
  if (!isPasswordValid) {
    throw new ApiError(401, "Invalid credentials");
  }

  // 5. Check OTP verification
  const verifiedTokenDoc = await Token.findOne({
    userId: user._id,
    tokenType: "verification",
    isVerified: true,
  });

  if (!verifiedTokenDoc) {
    // 👇 Generate new OTP and verification token
    const otp = generateOTP();
    const otpExpireAt = new Date(Date.now() + 10 * 60 * 1000); // 10 mins
    const verifyToken = jwt.sign(
      { userId: user._id },
      process.env.VERIFY_TOKEN_SECRET,
      { expiresIn: "15m" }
    );

    // ❌ Mark any previous unverified tokens as used
    await Token.updateMany(
      { userId: user._id, tokenType: "verification", isVerified: false, isUsed: false },
      { $set: { isUsed: true } }
    );

    // 🔁 Upsert (update or create) new token
    let tokenDoc = await Token.findOne({
      userId: user._id,
      tokenType: "verification",
      isVerified: false,
      isUsed: false
    });

    if (tokenDoc) {
      tokenDoc.token = verifyToken;
      tokenDoc.otp = otp;
      tokenDoc.otpExpireAt = otpExpireAt;
      tokenDoc.expiresAt = new Date(Date.now() + 15 * 60 * 1000);
      tokenDoc.isVerified = false;
      tokenDoc.isUsed = false;
      tokenDoc.otpType = "email";
      tokenDoc.authId = userAuth._id;
      await tokenDoc.save();
    } else {
      await Token.create({
        tokenType: "verification",
        token: verifyToken,
        otp,
        otpType: "email",
        otpExpireAt,
        userId: user._id,
        authId: userAuth._id,
        expiresAt: new Date(Date.now() + 15 * 60 * 1000),
        isVerified: false,
        isUsed: false
      });
    }

    // ✅ Send OTP to email
    await sendOTP(user.email, otp);

    // Return user info and verify token to prompt OTP
    return {
      requiresOTP: true,
      verifyToken,
      user: {
        fullName: user.fullName,
        email: user.email,
        phone: user.phone,
        userType: user.userType,
        isVerified: false,
      }
    };
  }

  // 6. Update last login if verified
  userAuth.lastLogin = new Date();
  await userAuth.save();

  // 7. Generate tokens
  const { accessToken, refreshToken } = await generateAccessAndRefreshTokens(
    user._id,
    null,
    {
      authId: userAuth._id,
      ipAddress: req.ip,
      userAgent: req.headers["user-agent"],
    }
  );

  // const safeUser = await User.findById(user._id).select("-__v");

  return {
    user: {
      fullName: user.fullName,
      email: user.email,
      phone: user.phone,
      userType: user.userType,
      isVerified: true,
    },
    accessToken,
    refreshToken
  };
};


export const verifyOTPService = async (req) => {
  const { otp } = req.body;

  const token = req.cookies.verify_token || req.headers["verificationtoken"];
  if (!token) {
    throw new ApiError(401, "Verification token missing");
  }

  let decoded;
  try {
    decoded = jwt.verify(token, process.env.VERIFY_TOKEN_SECRET);
  } catch (err) {
    throw new ApiError(401, "Invalid or expired verification token");
  }

  const userId = decoded.userId;
  const user = await User.findById(userId);
  if (!user) {
    throw new ApiError(404, "User not found");
  }

  const otpTokenDoc = await Token.findOne({
    userId,
    tokenType: "verification",
    isVerified: false,
    isUsed: false,
    otp,
  });

  if (
    !otpTokenDoc ||
    !otpTokenDoc.otp ||
    otpTokenDoc.otp !== otp ||
    new Date() > otpTokenDoc.otpExpireAt
  ) {
    throw new ApiError(400, "Invalid or expired OTP");
  }

  otpTokenDoc.isVerified = true;
  otpTokenDoc.isUsed = false;
  await otpTokenDoc.save({ validateBeforeSave: false });

  const userAuth = await UserAuth.findOne({ userId });
  if (!userAuth) {
    throw new ApiError(404, "User authentication record not found");
  }

  const { accessToken, refreshToken } = await generateAccessAndRefreshTokens(
    userId,
    null,
    {
      authId: userAuth._id,
      ipAddress: req.ip,
      userAgent: req.headers["user-agent"],
    }
  );

  const safeUser = await User.findById(userId).select("-__v");

  return { user: safeUser, accessToken, refreshToken };
};

export const logoutUserService = async (refreshToken) => {
  // console.log("Logout initiated with refresh token:", refreshToken);

  try {
    const deletedToken = await Token.findOneAndDelete({
      token: refreshToken,
      tokenType: "refresh",
    });

    if (!deletedToken) {
      console.warn("No matching refresh token document found in DB");
    } else {
      console.log("Refresh token deleted successfully from DB");
    }
  } catch (err) {
    console.error("Error during logout:", err);
    throw new ApiError(500, "Something went wrong during logout");
  }
};

export const refreshTokenService = async (incomingRefreshToken, req) => {
  let decoded;
  try {
    decoded = jwt.verify(incomingRefreshToken, process.env.REFRESH_TOKEN_SECRET);
    console.log("Decoded refresh token:", decoded);
  } catch (err) {
    throw new ApiError(401, "Invalid or expired refresh token");
  }

  // Step 1: Find token document
  const tokenDoc = await Token.findOne({
    token: incomingRefreshToken,
    tokenType: "refresh",
    isVerified: true,
    isUsed: false,
    expiresAt: { $gt: new Date() }
  });

  if (!tokenDoc) {
    throw new ApiError(403, "Invalid or expired refresh token");
  }

  // Step 2: Fetch user from tokenDoc
  const user = await User.findById(tokenDoc.userId);
  if (!user) {
    throw new ApiError(404, "User not found");
  }

  // Step 3: Mark current token as used
  tokenDoc.isUsed = true;
  await tokenDoc.save();

  // Step 4: Prepare meta for new token
  const meta = {
    ipAddress: req.ip,
    userAgent: req.headers["user-agent"]
  };

  // Step 5: Generate new tokens (reuse tokenDoc)
  const { accessToken, refreshToken } = await generateAccessAndRefreshTokens(user._id, tokenDoc, meta);

  return { accessToken, refreshToken };
};

export const resendOTPService = async (token) => {
  let decoded;

  try {
    decoded = jwt.verify(token, process.env.VERIFY_TOKEN_SECRET);
  } catch (err) {
    throw new ApiError(401, "Invalid or expired verification token");
  }

  const user = await User.findById(decoded.userId);
  if (!user) {
    throw new ApiError(404, "User not found");
  }

  const alreadyVerified = await Token.findOne({
    userId: user._id,
    tokenType: "verification",
    isVerified: true,
    isUsed: true
  });

  if (alreadyVerified) {
    throw new ApiError(400, "User already verified");
  }

  // Invalidate all previous unused verification tokens
  await Token.updateMany(
    { userId: user._id, tokenType: "verification", isVerified: false, isUsed: false },
    { $set: { isUsed: true } }
  );

  // Generate new OTP and JWT token
  const otp = generateOTP();
  const otpExpireAt = new Date(Date.now() + 10 * 60 * 1000); // 10 minutes

  const verifyToken = jwt.sign(
    { userId: user._id },
    process.env.VERIFY_TOKEN_SECRET,
    { expiresIn: "15m" }
  );

  await Token.create({
    tokenType: "verification",
    token: verifyToken,
    otp,
    otpType: "email", // You could pass this dynamically if needed
    otpExpireAt,
    userId: user._id,
    expiresAt: new Date(Date.now() + 15 * 60 * 1000),
    isVerified: false,
    isUsed: false
  });

  await sendOTP(user.email, otp);

  return { verifyToken, user };
};


    